export function setCookie(
  name: string,
  value: string,
  hours: number = 2
): void {
  const date = new Date();
  date.setTime(date.getTime() + hours * 60 * 60 * 1000);
  const expires = `expires=${date.toUTCString()}`;
  document.cookie = `${name}=${encodeURIComponent(value)};${expires};path=/`;
}

export function getCookie(name: string): string | null {
  const nameEQ = `${name}=`;
  const cookies = document.cookie.split(";");

  for (let i = 0; i < cookies.length; i++) {
    let cookie = cookies[i];
    while (cookie.charAt(0) === " ") {
      cookie = cookie.substring(1);
    }
    if (cookie.indexOf(nameEQ) === 0) {
      return decodeURIComponent(cookie.substring(nameEQ.length));
    }
  }
  return null;
}

export function deleteCookie(name: string): void {
  document.cookie = `${name}=;expires=Thu, 01 Jan 1970 00:00:00 GMT;path=/`;
}

export function getDismissedAnnouncements(): Set<string> {
  const dismissedCookie = getCookie("dismissedAnnouncements");
  if (!dismissedCookie) {
    return new Set();
  }

  try {
    const dismissedIds = JSON.parse(dismissedCookie);
    return new Set(dismissedIds);
  } catch (error) {
    console.error("Error parsing dismissed announcements cookie:", error);
    return new Set();
  }
}

export function saveDismissedAnnouncements(dismissedIds: Set<string>): void {
  const dismissedArray = Array.from(dismissedIds);
  setCookie("dismissedAnnouncements", JSON.stringify(dismissedArray), 2);
}

export function getDismissedOffers(): Set<string> {
  const dismissedCookie = getCookie("dismissedOffers");
  if (!dismissedCookie) {
    return new Set();
  }
  try {
    const dismissedIds = JSON.parse(dismissedCookie);
    return new Set(dismissedIds);
  } catch (error) {
    console.error("Error parsing dismissed offers cookie:", error);
    return new Set();
  }
}

export function saveDismissedOffers(dismissedIds: Set<string>): void {
  const dismissedArray = Array.from(dismissedIds);
  setCookie("dismissedOffers", JSON.stringify(dismissedArray), 2);
}
