import React from "react";
import {
  ThemeProvider,
  CssBaseline,
  createTheme,
  Box,
  Typography,
  useMediaQuery,
  useTheme,
  Avatar,
  Grow,
  Slide,
} from "@mui/material";

interface User {
  name: string;
  avatarUrl: string;
}

interface TrustedBusinessesProps {
  users?: User[];
  businessCount?: string;
  className?: string;
}

const TrustedBusinessesComponent: React.FC<TrustedBusinessesProps> = ({
  users = [
    {
      name: "Emma Johnson",
      avatarUrl:
        "https://images.pexels.com/photos/733872/pexels-photo-733872.jpeg?auto=compress&cs=tinysrgb&w=150",
    },
    {
      name: "Michael Chen",
      avatarUrl:
        "https://images.pexels.com/photos/2379005/pexels-photo-2379005.jpeg?auto=compress&cs=tinysrgb&w=150",
    },
    {
      name: "Amara Williams",
      avatarUrl:
        "https://images.pexels.com/photos/1239291/pexels-photo-1239291.jpeg?auto=compress&cs=tinysrgb&w=150",
    },
  ],
  businessCount = "30k+",
  className = "",
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));
  const isDarkMode = theme.palette.mode === "dark";

  const customTheme = createTheme({
    palette: {
      mode: isDarkMode ? "dark" : "light",
      primary: {
        main: "#3B82F6",
      },
      secondary: {
        main: "#14B8A6",
      },
      text: {
        primary: isDarkMode ? "#ffffff" : "#333333",
        secondary: isDarkMode
          ? "rgba(255, 255, 255, 0.7)"
          : "rgba(0, 0, 0, 0.6)",
      },
      background: {
        default: isDarkMode ? "#121212" : "#f8f9fa",
        paper: isDarkMode ? "#1e1e1e" : "#ffffff",
      },
      grey: {
        100: isDarkMode ? "#1e1e1e" : "#f5f5f5",
        900: isDarkMode ? "#121212" : "#212121",
      },
    },
    typography: {
      fontFamily: "Inter, sans-serif",
    },
    components: {
      MuiCssBaseline: {
        styleOverrides: `
        `,
      },
    },
  });

  const UserAvatar: React.FC<{ user: User; index: number; total: number }> = ({
    user,
    index,
    total,
  }) => {
    const marginLeft = index === 0 ? 0 : -3;
    const zIndex = total - index;

    return (
      <Box
        style={{
          right: "1rem",
          marginBottom: "-1rem",
          paddingBottom: "25px",
          marginTop: "-1rem",
        }}
        sx={{ marginLeft, zIndex, position: "relative" }}
      >
        <Grow in={true} timeout={index * 300}>
          <Avatar
            src={user.avatarUrl}
            alt={user.name}
            sx={{
              width: { xs: 48, sm: 56, md: 64 },
              height: { xs: 48, sm: 56, md: 64 },
              border: `2px solid ${customTheme.palette.background.paper}`,
              boxShadow: customTheme.shadows[2],
            }}
          />
        </Grow>
      </Box>
    );
  };

  return (
    <ThemeProvider theme={customTheme}>
      <CssBaseline />
      <Slide in={true} direction="up" timeout={500}>
        <Box className={className}>
          <Box
            sx={{
              display: "flex",
              flexDirection: isMobile ? "column" : "row",
              alignItems: "center",
              gap: isMobile ? 2 : 3,
              p: 2,
              backgroundColor: "background.default",
            }}
          >
            <Box sx={{ display: "flex", position: "relative" }}>
              {users.map((user, index) => (
                <UserAvatar
                  key={index}
                  user={user}
                  index={index}
                  total={users.length}
                />
              ))}
            </Box>

            <Typography
              style={{
                paddingBottom: "2rem",
                paddingRight: "5rem",
                textAlign: "center",
              }}
              component="span"
              sx={{
                fontFamily: '"Caveat", cursive',
                fontSize: { sm: "1rem", md: "1.2rem" },
                fontWeight: 500,
                color: "text.primary",
                ml: isMobile ? 6 : 0,
                mt: isMobile ? 1 : 0,
              }}
            >
              Trusted by {businessCount} Businesses
            </Typography>
          </Box>
        </Box>
      </Slide>
    </ThemeProvider>
  );
};

export default TrustedBusinessesComponent;
