"use client";

import React, { useState, useRef, useEffect } from "react";
import Image, { StaticImageData } from "next/image";
import { motion, AnimatePresence } from "framer-motion";
import { useTheme } from "../atoms/theme";

export interface TabItem {
  id: string | number;
  label: string;
  icon: React.ReactNode;
  lightImage: StaticImageData | string;
  darkImage: StaticImageData | string;
}

interface TabbedFeatureProps {
  tabs: TabItem[];
}

const TabbedFeature: React.FC<TabbedFeatureProps> = ({ tabs }) => {
  const [activeTab, setActiveTab] = useState(0);
  const { theme } = useTheme();
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const tabsRef = useRef<(HTMLButtonElement | null)[]>([]);
  const [manualChange, setManualChange] = useState(false);

  useEffect(() => {
    if (manualChange) return;

    const interval = setInterval(() => {
      setActiveTab((prev) => (prev + 1) % tabs.length);
    }, 3500);

    return () => clearInterval(interval);
  }, [manualChange, tabs.length]);

  useEffect(() => {
    if (scrollContainerRef.current && tabsRef.current[activeTab]) {
      const container = scrollContainerRef.current;
      const tab = tabsRef.current[activeTab];

      if (tab) {
        const containerRect = container.getBoundingClientRect();
        const tabRect = tab.getBoundingClientRect();
        const scrollLeft =
          container.scrollLeft +
          (tabRect.left - containerRect.left) -
          containerRect.width / 2 +
          tabRect.width / 2;

        container.scrollTo({
          left: scrollLeft,
          behavior: "smooth",
        });
      }
    }
  }, [activeTab]);

  return (
    <div className="w-full flex flex-col items-center">
      <div className="relative w-full max-w-[1100px] flex items-center justify-center ">
        <div
          className="pointer-events-none absolute left-0 top-0 h-full w-16 z-20 
    bg-gradient-to-r from-white dark:from-[#1a1a1a] to-transparent"
        />

        <div
          className="pointer-events-none absolute right-0 top-0 h-full w-16 z-20 
    bg-gradient-to-l from-white dark:from-[#1a1a1a] to-transparent"
        />

        <div
          ref={scrollContainerRef}
          className="overflow-x-auto [&::-webkit-scrollbar]:hidden 
               [-ms-overflow-style:'none'] [scrollbar-width:none]"
        >
          <div className="flex min-w-max gap-4 px-4 mb-2 md:mb-4 justify-center">
            {tabs.map((tab, index) => (
              <button
                ref={(el) => {
                  tabsRef.current[index] = el;
                }}
                key={tab.id}
                onClick={() => {
                  setActiveTab(index);
                  setManualChange(true);
                }}
                className={`flex flex-col items-center gap-2 px-4 py-0 text-sm font-medium
    transition-all whitespace-nowrap
    ${
      activeTab === index
        ? "text-[#4683ff]"
        : "text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
    }`}
              >
                {tab.icon}
                <span className="text-xs font-light">{tab.label}</span>
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="relative w-full max-w-[1233px]">
        <div className="relative w-full aspect-[16/10] sm:aspect-[16/9] md:aspect-[1100/600] rounded-2xl bg-gray-100 dark:bg-[#1a1a1a] overflow-hidden shadow-sm">
          <Image
            src={theme === "dark" ? tabs[0].darkImage : tabs[0].lightImage}
            alt="Spacer"
            className="invisible w-full h-full object-cover"
            width={1100}
            height={700}
            priority
          />

          <AnimatePresence mode="wait">
            <motion.div
              key={activeTab}
              initial={{ opacity: 0, y: 10 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -10 }}
              transition={{ duration: 0.3 }}
              className="absolute inset-0 w-full h-full"
            >
              <Image
                src={
                  theme === "dark"
                    ? tabs[activeTab].darkImage
                    : tabs[activeTab].lightImage
                }
                alt={tabs[activeTab].label}
                className="w-full h-full rounded-2xl"
                width={1300}
                height={700}
                priority
              />
            </motion.div>
          </AnimatePresence>
        </div>
      </div>
    </div>
  );
};

export default TabbedFeature;
