"use client";
import React, { useState, useEffect, useRef } from "react";
import Image from "next/image";
import type { StaticImageData } from "next/image";
import { useTheme } from "../atoms/theme";

import crmdark from "../../public/images/crmdark.png";
import crmlight from "../../public/images/crmlight.png";
import hrdark from "../../public/images/hrdark.png";
import hrlight from "../../public/images/hrlight.png";
import campaignsdark from "../../public/images/campaignsdark.png";
import campaignslight from "../../public/images/campaignslight.png";
import chatdark from "../../public/images/chatdark.png";
import chatlight from "../../public/images/chatlight.png";
import workdark from "../../public/images/workdark.png";
import worklight from "../../public/images/worklight.png";
import CrewPic from "../../public/images/clikkleCrew.png";

interface Product {
  id: string;
  name: string;
  logo: string;
  screenshot: {
    light: string | StaticImageData;
    dark: string | StaticImageData;
  };
}

const products: Product[] = [
  {
    id: "worksuite",
    name: "Worksuite",
    logo: "https://cdn.clikkle.com/images/worksuite/2025/clikkle%20worksuite.png",
    screenshot: { light: worklight, dark: workdark },
  },
  {
    id: "chat",
    name: "Chat",
    logo: "https://cdn.clikkle.com/images/chat/logo/2025/clikkle%20chat.png",
    screenshot: { light: chatlight, dark: chatdark },
  },
  {
    id: "files",
    name: "Files",
    logo: "https://cdn.clikkle.com/images/files/logo/2025/clikkle%20files.png",
    screenshot: {
      light: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/files/files%20dedicated%20app/activity%20report%20-daymode.png",
      dark: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/files/files%20dedicated%20app/activity%20report%20-darkmode.png",
    },
  },
  {
    id: "esign",
    name: "eSign",
    logo: "https://cdn.clikkle.com/images/e-sign/logo/2025/clikkle%20esign.png",
    screenshot: {
      light: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/esign/esign%20dedicated%20app/esign%20app-daymode.png",
      dark: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/esign/esign%20dedicated%20app/esign%20app-darkmode.png",
    },
  },
  {
    id: "hr",
    name: "HR",
    logo: "https://cdn.clikkle.com/images/hr/logo/2025/clikkle%20hr.png",
    screenshot: { light: hrlight, dark: hrdark },
  },
  {
    id: "projects",
    name: "Projects",
    logo: "https://cdn.clikkle.com/images/projects/logo/2025/clikkle%20projects.png",
    screenshot: {
      light: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/projects/dedicated%20project%20app/project%20dash-daymode.png",
      dark: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/projects/dedicated%20project%20app/project%20dash-darkmode.png",
    },
  },
  {
    id: "campaigns",
    name: "Campaigns",
    logo: "https://cdn.clikkle.com/images/campaigns/logo/2025/clikkle%20campaigns.png",
    screenshot: { light: campaignslight, dark: campaignsdark },
  },
  {
    id: "crm",
    name: "CRM",
    logo: "https://cdn.clikkle.com/images/crm/2025/clikkle%20crm.png",
    screenshot: { light: crmlight, dark: crmdark },
  },
  {
    id: "crew",
    name: "Crew",
    logo: "https://cdn.clikkle.com/images/crew/logo/2025/Clikkle%20Crew.png",
    screenshot: { light: CrewPic, dark: CrewPic },
  },
  {
    id: "calendar",
    name: "Calendar",
    logo: "https://cdn.clikkle.com/images/calendar/logo/2025/Clikkle%20Calendar.png",
    screenshot: {
      light: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/calendar/dedicated%20app/calendar%20availability-daymode.png",
      dark: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/calendar/dedicated%20app/calendar%20availability-darkmode.png",
    },
  },
  {
    id: "creatorai",
    name: "CreatorAI",
    logo: "https://cdn.clikkle.com/images/creator/logo/2025/Clikkle%20Creator.png",
    screenshot: {
      light: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/creator%20ai/image_01%20(2).png",
      dark: "https://cdn.clikkle.com/SCREEN%20SHOTS/CLIKKLE/creator%20ai/image_01%20(2).png",
    },
  },
];

export const MacWindowCarousel: React.FC<{ initialProduct?: string }> = ({ initialProduct }) => {
  const { theme } = useTheme();

  const getInitialIndex = () => {
    if (initialProduct) {
      const index = products.findIndex((p) => p.id.toLowerCase() === initialProduct.toLowerCase());
      return index !== -1 ? index : 0;
    }
    return 0;
  };

  const [activeIndex, setActiveIndex] = useState(getInitialIndex);
  const [isPaused, setIsPaused] = useState(false);
  const intervalRef = useRef<ReturnType<typeof setInterval> | null>(null);

  useEffect(() => {
    if (initialProduct || isPaused) {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
        intervalRef.current = null;
      }
      return;
    }

    intervalRef.current = setInterval(() => {
      setActiveIndex((prev) => (prev + 1) % products.length);
    }, 3500);

    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
        intervalRef.current = null;
      }
    };
  }, [initialProduct, isPaused]);

  const handleIconClick = (index: number) => {
    setActiveIndex(index);
    setIsPaused(true);
    if (intervalRef.current) {
      clearInterval(intervalRef.current);
      intervalRef.current = null;
    }
  };

  const activeProduct = products[activeIndex];
  const borderColor = theme === "dark" ? "#333333" : "#e5e7eb";
  const activeTabBg = theme === "dark" ? "#141414" : "#ffffff";
  const patternDotFill = theme === "dark" ? "#333333" : "#d1d5db";
  const headerBg = theme === "dark" ? "#141414" : "#f3f4f6";

  return (
    <div className="flex items-center justify-center pb-8 rounded-lg w-full">
      <div
        className="relative w-full max-w-[1200px] overflow-hidden rounded-[16px] border shadow-2xl transition-colors duration-300"
        style={{ borderColor: borderColor, backgroundColor: headerBg }}
      >
        <div className="relative w-full">
          <svg className="absolute top-0 left-0 z-0 h-full w-full opacity-40 dark:opacity-80 pointer-events-none">
            <pattern id="pattern-dots" x="0" y="0" width="6.5" height="10" patternUnits="userSpaceOnUse">
              <circle cx="2.5" cy="2.5" r="0.75" fill={patternDotFill}></circle>
              <circle cx="5" cy="7.5" r="0.75" fill={patternDotFill}></circle>
            </pattern>
            <rect x="0" y="0" width="100%" height="100%" fill="url(#pattern-dots)"></rect>
          </svg>

          <div
            className="relative flex items-end justify-start"
            style={{
              borderTopColor: borderColor,
              borderRightColor: borderColor,
              borderLeftColor: borderColor,
              borderBottom: "0px"
            }}
          >            <div className="flex items-center space-x-1 px-3 sm:px-5 pb-3.5 pt-3.5 z-20">
              <div className="w-2 h-2 sm:w-3 sm:h-3 rounded-full bg-[#2f56b7] border border-[#2f56b7]"></div>
              <div className="w-2 h-2 sm:w-3 sm:h-3 rounded-full bg-[#4072f5] border border-[#4072f5]"></div>
              <div className="w-2 h-2 sm:w-3 sm:h-3 rounded-full bg-[#aec6fc] border border-[#aec6fc]"></div>
            </div>

            <div className="flex h-full w-full grow items-center overflow-hidden">
              <div className="flex items-end pt-2 relative w-full overflow-x-auto no-scrollbar">

                <div
                  className="relative z-10 shrink-0 transition-all duration-500 ease-in-out w-[120px] sm:w-[200px] h-9 sm:h-[42px]"
                >
                  <svg width="100%" height="100%" viewBox="0 0 200 42" preserveAspectRatio="none" fill="none" className="absolute inset-0">
                    <path d="M16.1689 0.5H184.033C188.175 0.500002 191.533 3.85787 191.533 8V34C191.533 38 195 41.5 200 41.5H0C5 41.5 8.669 38 8.669 34V8C8.669 3.85787 12.0268 0.5 16.1689 0.5Z" fill={activeTabBg} />
                    <path d="M16.1689 0.5H184.033C188.175 0.500002 191.533 3.85787 191.533 8V34C191.533 38 195 41.5 200 41.5" stroke={borderColor} fill="none" />
                    <path d="M16.1689 0.5C12.0268 0.5 8.669 3.85787 8.669 8V34C8.669 38 5 41.5 0 41.5" stroke={borderColor} fill="none" />
                  </svg>

                  <div className="relative z-10 flex h-full w-full items-center justify-start gap-1 sm:gap-2 pl-3 sm:pl-5 pr-2 select-none">
                    <Image src={activeProduct.logo} alt={activeProduct.name} width={14} height={14} className="w-2.5 h-2.5 sm:w-3.5 sm:h-3.5 object-contain" />
                    <span className={`text-[10px] sm:text-sm font-medium truncate ${theme === 'dark' ? 'text-white' : 'text-gray-900'}`}>
                      {activeProduct.name}
                    </span>
                  </div>
                </div>

                <div className="flex items-center pb-2 ml-1 transition-all duration-500 h-9 sm:h-[42px] flex-nowrap">
                  {products.map((product, index) => {
                    const isHidden = index === activeIndex;
                    return (
                      <div
                        key={product.id}
                        className={`flex items-center transition-all duration-500 justify-center ease-in-out ${isHidden ? "w-0 opacity-0 invisible" : "w-[45px] sm:w-[65px]"} shrink-0 overflow-hidden`}
                      >
                        <div className="relative flex items-center justify-center px-2 sm:px-4" style={{ minWidth: isHidden ? undefined : '44px' }}>
                          <button
                            onClick={() => handleIconClick(index)}
                            className="cursor-pointer group flex items-center justify-center transition-all duration-200 hover:scale-110"
                            aria-label={`Show ${product.name}`}
                          >
                            <Image src={product.logo} alt={product.name} width={14} height={14} className="w-2.5 h-2.5 sm:w-3.5 sm:h-3.5 object-contain" />
                          </button>
                        </div>
                        {!isHidden && <div className="h-3 sm:h-4 w-px shrink-0" style={{ backgroundColor: borderColor }}></div>}
                      </div>
                    );
                  })}
                </div>
              </div>
            </div>
          </div>
        </div>

        <div className="relative aspect-video w-full overflow-hidden" style={{ backgroundColor: activeTabBg }}>
          {products.map((product, index) => {
            const imgSrc = theme === "dark" ? product.screenshot.dark : product.screenshot.light;
            return (
              <div
                key={product.id}
                className={`absolute inset-0 transition-opacity duration-700 ease-in-out ${index === activeIndex ? "opacity-100 z-10" : "opacity-0 z-0"}`}
              >
                {typeof imgSrc === 'string' ? (
                  
                  <img src={imgSrc} alt={product.name} className="w-full h-full object-cover object-top" />
                ) : (
                  <Image src={imgSrc} alt={product.name} className="w-full h-full object-cover object-top" priority={index === activeIndex} />
                )}
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
};
