"use client";

import { useState } from "react";
import { X } from "lucide-react";
import accountingIcon from "../../public/images/quickbooks.svg";
import ecommerceIcon from "../../public/images/shopify.svg";
import crmIcon from "../../public/images/salesforce.svg";
import chatIcon from "../../public/images/slack.svg";
import hrIcon from "../../public/images/bamboohr.svg";
import signIcon from "../../public/images/docusign.svg";
import inventoryIcon from "../../public/images/inflow.svg";
import documentsIcon from "../../public/images/sharepoint.svg";
import posIcon from "../../public/images/lightspeed.svg";
import expensesIcon from "../../public/images/expensify.svg";
import appointmentIcon from "../../public/images/calendly.svg";
import purchaseIcon from "../../public/images/coupa.svg";
import socialIcon from "../../public/images/hootsuite.svg";
import planningIcon from "../../public/images/asana.svg";
import emailingIcon from "../../public/images/google.svg";
import timesheetIcon from "../../public/images/harvest.svg";
import helpdeskIcon from "../../public/images/zendesk.svg";
import eventsIcon from "../../public/images/eventbrite.svg";
import mrpIcon from "../../public/images/katanamrp.svg";
import Image from "next/image";

interface App {
  id: number;
  name: string;
  icon: string;
  price: number;
}

const apps: App[] = [
  {
    id: 2,
    name: "Accounting",
    icon: accountingIcon,
    price: 20,
  },
  {
    id: 3,
    name: "eCommerce",
    icon: ecommerceIcon,
    price: 20,
  },
  {
    id: 4,
    name: "CRM",
    icon: crmIcon,
    price: 20,
  },
  {
    id: 5,
    name: "Discuss",
    icon: chatIcon,
    price: 20,
  },
  {
    id: 7,
    name: "HR",
    icon: hrIcon,
    price: 20,
  },
  {
    id: 8,
    name: "Sign",
    icon: signIcon,
    price: 20,
  },
  {
    id: 9,
    name: "Inventory",
    icon: inventoryIcon,
    price: 20,
  },
  {
    id: 11,
    name: "Documents",
    icon: documentsIcon,
    price: 20,
  },
  {
    id: 12,
    name: "PoS",
    icon: posIcon,
    price: 20,
  },
  {
    id: 13,
    name: "Expenses",
    icon: expensesIcon,
    price: 20,
  },
  {
    id: 14,
    name: "Appointment",
    icon: appointmentIcon,
    price: 20,
  },
  {
    id: 15,
    name: "Purchase",
    icon: purchaseIcon,
    price: 20,
  },
  {
    id: 16,
    name: "Social",
    icon: socialIcon,
    price: 20,
  },
  {
    id: 17,
    name: "Planning",
    icon: planningIcon,
    price: 20,
  },
  {
    id: 18,
    name: "Emailing",
    icon: emailingIcon,
    price: 20,
  },
  {
    id: 19,
    name: "Timesheet",
    icon: timesheetIcon,
    price: 20,
  },
  {
    id: 20,
    name: "Helpdesk",
    icon: helpdeskIcon,
    price: 20,
  },
  {
    id: 21,
    name: "Events",
    icon: eventsIcon,
    price: 20,
  },
  {
    id: 22,
    name: "MRP",
    icon: mrpIcon,
    price: 20,
  },
];

const ODOO_MONTHLY_PER_USER = 2;

export default function PricingCalculator() {
  const [selectedApps, setSelectedApps] = useState<Record<number, boolean>>(
    () => {
      const defaultSelectedApps: Record<number, boolean> = {};
      for (let i = 1; i <= 7; i++) {
        defaultSelectedApps[i] = true;
      }
      return defaultSelectedApps;
    }
  );
  const [users, setUsers] = useState<number>(10);

  const toggleApp = (id: number) => {
    setSelectedApps((prev) => ({
      ...prev,
      [id]: !prev[id],
    }));
  };

  const removeApp = (id: number) => {
    setSelectedApps((prev) => {
      const next = { ...prev };
      delete next[id];
      return next;
    });
  };

  const replacedMonthlyPerUser = apps
    .filter((app) => selectedApps[app.id])
    .reduce((sum, app) => sum + app.price, 0);

  const replacedMonthlyTotal = replacedMonthlyPerUser * users;
  const replacedYearlyTotal = replacedMonthlyTotal * 12;

  const odooMonthlyTotal = ODOO_MONTHLY_PER_USER * users;
  const odooYearlyTotal = odooMonthlyTotal * 12;

  const savingsYearly = Math.max(0, replacedYearlyTotal - odooYearlyTotal);

  return (
    <section className="py-16 bg-white dark:bg-[#1a1a1a] max-w-7xl mx-auto">
      <div className="container mx-auto px-4">
        <h2 className="text-3xl md:text-5xl font-black text-center mb-8 text-[#286efb] dark:text-[#A9C5FF]">
          Save more with Clikkle
        </h2>
        <p className="text-center text-gray-600 dark:text-[#d7d7de] mb-8">
          Cost savings based on average price per user for each app.
        </p>

        <div className="flex flex-col lg:flex-row gap-8">
          <div className="flex-1 bg-gray-50 dark:bg-[#141414] rounded-2xl p-6 shadow-sm border border-gray-200 dark:border-[#5e5e69]">
            <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
              Which apps do you use?
            </h3>
            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-4">
              {apps.map((app) => (
                <div
                  key={app.id}
                  role="button"
                  onClick={() => toggleApp(app.id)}
                  className={`relative cursor-pointer border rounded-2xl p-4 flex flex-col items-center justify-center transition-all duration-200 hover:shadow-lg ${
                    selectedApps[app.id]
                      ? "border-[#286efb] dark:border-[#ffffff] bg-blue-50 dark:bg-[#141414] shadow-md"
                      : "border-gray-200 dark:border-[#5e5e69] bg-white dark:bg-[#1a1a1a] hover:border-gray-300 dark:hover:border-[#7a7a85] hover:shadow-sm"
                  }`}
                >
                  {selectedApps[app.id] && (
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        removeApp(app.id);
                      }}
                      className="absolute top-2 right-2 p-1 rounded-md hover:bg-gray-100 dark:hover:bg-[#3a3a45] transition-colors"
                      aria-label={`Remove ${app.name}`}
                    >
                      <X className="w-4 h-4 text-gray-600 dark:text-gray-300" />
                    </button>
                  )}
                  <div className="mb-2">
                    <Image
                      src={app.icon}
                      alt={`${app.name} icon`}
                      className="w-6 h-6 object-contain"
                      width={24}
                      height={24}
                    />
                  </div>
                  <span className="font-medium text-sm text-gray-900 dark:text-white text-center">
                    {app.name}
                  </span>
                  <div className="text-xs text-gray-600 dark:text-gray-400 mt-2">
                    {selectedApps[app.id]
                      ? `$${app.price}/user`
                      : "Tap to select"}
                  </div>
                </div>
              ))}
            </div>
            <div className="mt-6 flex items-center gap-4">
              <div className="flex items-center gap-2 bg-white dark:bg-[#1a1a1a] rounded-md p-2 border border-gray-200 dark:border-[#5e5e69]">
                <button
                  onClick={() => setUsers((u) => Math.max(1, u - 1))}
                  className="w-10 h-10 rounded-md border border-gray-300 dark:border-[#5e5e69] bg-white dark:bg-[#141414] text-gray-900 dark:text-white hover:bg-gray-50 dark:hover:bg-[#3a3a45] transition-colors"
                >
                  —
                </button>
                <div className="w-20 text-center font-medium text-gray-900 dark:text-white">
                  {users}
                </div>
                <button
                  onClick={() => setUsers((u) => Math.min(999, u + 1))}
                  className="w-10 h-10 rounded-md border border-gray-300 dark:border-[#5e5e69] bg-white dark:bg-[#141414] text-gray-900 dark:text-white hover:bg-gray-50 dark:hover:bg-[#3a3a45] transition-colors"
                >
                  +
                </button>
              </div>
              <div className="ml-auto text-sm text-gray-600 dark:text-[#d7d7de]">
                <div>How many users?</div>
              </div>
            </div>
          </div>
          <aside className="w-full lg:w-96 bg-gray-50 dark:bg-[#141414] rounded-2xl p-6 shadow-sm border border-gray-200 dark:border-[#5e5e69]">
            <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
              Apps to replace for {users} users
            </h3>
            <div className="divide-y divide-gray-200 dark:divide-[#5e5e69]">
              {Object.keys(selectedApps).length === 0 && (
                <div className="py-4 text-sm text-gray-600 dark:text-gray-400">
                  No apps selected
                </div>
              )}
              {apps
                .filter((app) => selectedApps[app.id])
                .map((app) => (
                  <div
                    key={app.id}
                    className="flex items-center justify-between py-3 text-sm"
                  >
                    <div className="flex items-center gap-3">
                      <div className="w-8 h-8 rounded-md bg-white dark:bg-[#1a1a1a] border border-gray-200 dark:border-[#5e5e69] flex items-center justify-center">
                        <Image
                          src={app.icon}
                          alt={`${app.name} icon`}
                          className="w-5 h-5 object-contain"
                          width={20}
                          height={20}
                        />
                      </div>
                      <div>
                        <div className="font-medium text-gray-900 dark:text-white">
                          {app.name}
                        </div>
                      </div>
                    </div>
                    <div className="text-gray-600 dark:text-gray-400">
                      ${app.price.toFixed(2)} / user
                    </div>
                  </div>
                ))}
              <div className="py-3 flex items-center justify-between border-t border-gray-200 dark:border-[#5e5e69]">
                <div className="font-medium text-gray-900 dark:text-white">
                  TOTAL
                </div>
                <div className="font-semibold text-gray-900 dark:text-white">
                  ${(replacedMonthlyPerUser * users * 12).toFixed(2)} / year
                </div>
              </div>
            </div>
            <hr className="my-4 border-gray-200 dark:border-[#5e5e69]" />
            <div className="text-sm text-gray-600 dark:text-[#d7d7de]">
              <div>
                All Clikkle Apps (without the &apos;corporate&apos; tax)
              </div>
              <div className="mt-2 font-semibold text-gray-900 dark:text-white">
                ${odooYearlyTotal.toFixed(2)} / year
              </div>
            </div>
            <div className="mt-6 bg-[#286efb] rounded-xl p-4 text-center">
              <div className="text-sm text-white/90">Your savings</div>
              <div className="text-2xl font-bold text-white">
                ${savingsYearly.toFixed(2)} / year
              </div>
              <div className="text-xs mt-1 text-white/80">
                For a fully-integrated software. (No Corporate BS)
              </div>
            </div>
          </aside>
        </div>
      </div>
    </section>
  );
}
