"use client";
/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect } from "react";
import { Box, IconButton } from "@mui/material";
import { usePathname } from "next/navigation";
import CloseIcon from "@mui/icons-material/Close";

import DOMPurify from "dompurify";

const DEFAULT_API_BASE_URL = "https://admin.clikkle.com:5100/api";

const AnnouncementBar = ({
  apiBaseUrl = DEFAULT_API_BASE_URL,
  onClose = () => {},
}: any) => {
  const [announcements, setAnnouncements] = useState([]);
  const [dismissedIds, setDismissedIds] = useState(new Set());
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);
  const pathname = usePathname();
  const pageName = pathname === "/" ? "/" : `/${pathname?.split("/")[1] || ""}`;

  useEffect(() => {
    const fetchAnnouncements = async () => {
      try {
        setIsLoading(true);
        setError(null);

        const response = await fetch(
          `${apiBaseUrl}/announcements/active?pageName=${pageName}`
        );

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();
        setAnnouncements(data?.data || []);
      } catch (err: any) {
        console.error("Error fetching announcements:", err);
        setError(err.message);
        setAnnouncements([]);
      } finally {
        setIsLoading(false);
      }
    };

    fetchAnnouncements();

    const interval = setInterval(fetchAnnouncements, 5 * 60 * 1000);
    return () => clearInterval(interval);
  }, [pageName, apiBaseUrl]);

  const visibleAnnouncements = announcements?.filter(
    (announcement: any) => !dismissedIds.has(announcement._id)
  );

  const handleDismiss = (announcementId: any) => {
    const newDismissed = new Set(dismissedIds);
    newDismissed.add(announcementId);
    setDismissedIds(newDismissed);

    if (onClose) {
      onClose(announcementId);
    }
  };

  const handleClick = (announcement: any) => {
    if (announcement.onClick) {
      try {
        const func = new Function(announcement.onClick);
        func();
      } catch (error) {
        console.error("Error executing onClick:", error);
      }
    }
  };

  if (isLoading || error || visibleAnnouncements.length === 0) {
    return null;
  }

  return (
    <Box>
      {visibleAnnouncements?.map((announcement: any) => (
        <Box
          key={announcement._id}
          onClick={() => announcement.onClick && handleClick(announcement)}
          sx={{
            backgroundColor: announcement.backgroundColor || "black",
            color: "white",
            display: "flex",
            alignItems: "center",
            justifyContent: { xs: "center", md: "center" },
            gap: 1,
            py: "0.5rem",
            position: "relative",
            px: 2,
            fontWeight: 600,
            cursor: announcement.onClick ? "pointer" : "default",
            "&:hover": announcement.onClick
              ? { filter: "brightness(0.9)" }
              : {},
          }}
          role="button"
          aria-label="announcement"
        >
          <Box
            component="div"
            sx={{
              textAlign: { xs: "left", md: "center" },
              fontSize: { xs: "10px", md: "14px" },
              lineHeight: 1.3,
              width: "auto" ,
              display: "flex",
              justifyContent: { xs: "center", md: "center" },
              alignItems: { xs: "center", md: "center" },
              gap: 1,
            }}
          >
            {announcement.text && (
              <>
                <span
                  data-announcement
                  className="prose prose-neutral text-center my-1"
                  dangerouslySetInnerHTML={{
                    __html: DOMPurify.sanitize(announcement.text),
                  }}
                />
              </>
            )}
          </Box>

          <IconButton
            aria-label="dismiss announcement"
            onClick={(e) => {
              e.stopPropagation();
              handleDismiss(announcement._id);
            }}
            sx={{
              position: "absolute",
              right: { xs: "10px", md: "120px" },
              top: "50%",
              transform: "translateY(-50%)",
              color: "#fff",
              padding: 0.3,
              backgroundColor: "#343a40",
            }}
            size="small"
          >
            <CloseIcon fontSize="small" />
          </IconButton>
        </Box>
      ))}
    </Box>
  );
};

export default AnnouncementBar;
