/* eslint-disable @next/next/no-img-element */
"use client";
/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useRef } from "react";
import Link from "next/link";
import Image from "next/image";
import { useTheme } from "../atoms/theme";
import {
  Search,
  Calendar,
  Clock,
  Filter,
  SortAsc,
  SortDesc,
  ArrowRightIcon,
  Logs,
  Ellipsis,
} from "lucide-react";
import { Button } from "../components/button";
import { Card, CardContent } from "../components/card";
import { BlogPost, blogCategories } from "../constants/blogData";
import ThemeSwitch from "../components/themeSwitch";
import { FaArrowRight } from "react-icons/fa";
import { footerColumns } from "../constants/footerText";
import { apps } from "../constants/products";
import illustration from "../../public/images/banner.png";
import SidebarDrawer from "../components/sidebarDrawer";
import AnnouncementBar from "../components/AnnouncementBar";
import { getCookie } from "../utils/cookies";
import ClikkleLogo from "@/components/ClikkleLogo";
import Footer from "@/components/Footer";
import { FaArrowCircleDown } from "react-icons/fa";
import { getEncodedReferrer } from "@/utils/utils";

const DEFAULT_API_BASE_URL = "https://admin.clikkle.com:5100/api";

const POSTS_PER_PAGE = 6;

const API_BASE_URL = DEFAULT_API_BASE_URL;

const BlogsHome = () => {
  const { theme } = useTheme();
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("All");
  const [sortBy, setSortBy] = useState<"date" | "title" | "order">("date");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [currentPage, setCurrentPage] = useState(1);
  const [posts, setPosts] = useState<BlogPost[]>([]);
  const [totalCount, setTotalCount] = useState(0);
  const [totalPages, setTotalPages] = useState(1);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [isVisible] = useState(false);

  useEffect(() => {
    const controller = new AbortController();
    const fetchBlogs = async () => {
      try {
        setLoading(true);
        setError(null);
        const params = new URLSearchParams();
        params.set("page", String(currentPage));
        params.set("limit", String(POSTS_PER_PAGE));
        params.set("status", "published");
        if (searchTerm) params.set("search", searchTerm);
        if (selectedCategory !== "All")
          params.set("category", selectedCategory);
        const sortByServer =
          sortBy === "date"
            ? "publishedDate"
            : sortBy === "order"
            ? "orderIndex"
            : "title";
        params.set("sortBy", sortByServer);
        params.set("sortOrder", sortOrder);

        const res = await fetch(
          `${API_BASE_URL}/blogs/public?${params?.toString()}`,
          {
            signal: controller.signal,
          }
        );
        const json = await res.json();
        if (!res.ok || !json?.success) {
          throw new Error(json?.error || "Failed to fetch blogs");
        }
        const apiBlogs = Array.isArray(json.data) ? json.data : [];
        const mapped: BlogPost[] = apiBlogs.map((b: any) => ({
          id: b._id,
          slug: b.slug,
          title: b.title,
          excerpt: b.excerpt || "",
          content: b.content || "",
          featuredImage:
            b.featuredImage ||
            "https://cdn.clikkle.com/images/worksuite/2025/clikkle%20worksuite.png",
          author: {
            name: b.author?.name || "Unknown Author",
            avatar:
              b.author?.avatar ||
              "https://cdn.clikkle.com/images/worksuite/2025/clikkle%20worksuite.png",
            bio: b.author?.bio || "",
          },
          publishedDate: b.publishedDate || new Date().toISOString(),
          category: b.category || "News",
          tags: Array.isArray(b.tags) ? b.tags : [],
          readTime: b.readTime ?? b.calculatedReadTime ?? 0,
          seoTitle: b.metaTitle || b.title,
          seoDescription: b.metaDescription || b.excerpt || "",
        }));
        if (currentPage === 1) {
          setPosts(mapped);
        } else {
          setPosts((prev) => [...prev, ...mapped]);
        }
        setTotalCount(json.pagination?.total ?? mapped.length);
        setTotalPages(json.pagination?.pages ?? 1);
      } catch (err: any) {
        if (err?.name === "AbortError") return;
        setError(err?.message || "Unexpected error fetching blogs");
      } finally {
        if (!controller.signal.aborted) {
          setLoading(false);
        }
      }
    };

    fetchBlogs();
    return () => controller.abort();
  }, [searchTerm, selectedCategory, sortBy, sortOrder, currentPage]);

  useEffect(() => {
    setCurrentPage(1);
  }, [searchTerm, selectedCategory, sortBy, sortOrder]);

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "long",
      day: "numeric",
    });
  };

  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);

  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        productsRef.current &&
        !productsRef.current.contains(event.target as Node)
      ) {
        setProductsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);
  const [announcementOpen, setAnnouncementOpen] = useState(false);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (el && el.offsetHeight > 0) {
        setAnnouncementOpen(true);
      } else {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();

    const interval = setInterval(checkDismissedAnnouncements, 1000);

    return () => clearInterval(interval);
  }, []);
  return (
    <>
      <div className={`min-h-screen bg-white dark:bg-[#1a1a1a]`}>
        <section className="w-full">
          <section className="relative w-full h-28 overflow-hidden">
            <div
              className={`fixed top-0 inset-x-0 z-50 ${
                scrolled
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : productsOpen || dotsOpen
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : ""
              }`}
            >
              <div
                ref={(el) => {
                  if (el) {
                    const height = el.getBoundingClientRect().height;

                    if (height !== 0) {
                      setAnnouncementOpen(true);
                    }
                  }
                }}
              >
                <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
              </div>
              <div className="container max-w-[1233px] mx-auto max-[1300px]:px-14 max-[1100px]:px-4">
                <nav
                  className={`relative flex items-center justify-between py-4`}
                >
                  <ClikkleLogo />

                  <div className="flex md:hidden items-center">
                    <div
                      className="flex items-center gap-0 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-[8px] shadow-sm"
                      onClick={toggleLeftDrawer}
                    >
                      <p className="text-sm font-medium text-gray-900 dark:text-white">
                        <Logs className="w-5 h-5 text-gray-900 dark:text-white mr-3" />
                      </p>

                      <p className="flex items-center  rounded-xl text-sm font-bold ">
                        Menu
                      </p>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-12 font-inter font-medium text-[18px]">
                    <Link
                      href="https://clikkle.com#features"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Features
                    </Link>
                    <Link
                      href="https://clikkle.com/#pricing"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Pricing
                    </Link>
                    <Link
                      href="/blogs"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Blog
                    </Link>
                    <div className="flex flex-row justify-center items-center">
                      <div className="relative" ref={productsRef}>
                        <button
                          onClick={() => setProductsOpen((o) => !o)}
                          className="cursor-pointer flex items-center gap-1 text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                        >
                          Products
                        </button>
                        {productsOpen && (
                          <div
                            className={`fixed ${
                              announcementOpen ? "top-[115px]" : "top-[75px]"
                            } 
              left-1/2 transform -translate-x-1/2 w-full bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
              shadow-lg p-6 border-b-2 border-gray-300 dark:border-[#575662] z-50 flex justify-center items-center pb-0`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                      <div className="flex flex-col gap-3">
                              {apps
                                .slice(0, Math.ceil(apps.length / 2))
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                            </div>

                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(
                                  Math.ceil(apps.length / 2),
                                  Math.ceil(apps.length / 2) + 2,
                                )
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                              <Link href="/products">
                                <Button
                                  style={{ marginTop: "5px" }}
                                  variant="ghost"
                                  className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                >
                                  View all products
                                  <ArrowRightIcon className="h-2 w-2 dark:text-white" />
                                </Button>
                              </Link>
                            </div>
                              <div className="hidden lg:flex flex-row items-center justify-center text-center gap-3">
                                <Image
                                  src={illustration}
                                  alt="Illustration"
                                  className="w-auto h-[175px]"
                                  width={400}
                                  height={175}
                                />
                                <div className="flex flex-col justify-center items-center h-full">
                                  <p className="text-sm text-gray-700 dark:text-gray-100 max-w-[200px] pb-5">
                                    Get started today the modern solution to
                                    power your business.
                                  </p>
                                  <Button
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className="cursor-pointer w-44 font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto border border-solid"
                                  >
                                    Get Started Today
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>

                      <div className="relative" ref={dotsRef}>
                        <Ellipsis
                          onClick={() => setDotsOpen((o) => !o)}
                          className="w-8 h-6 rounded-full p-[1px] cursor-pointer"
                        />
                        {dotsOpen && (
                          <div
                            className={`fixed ${
                              isVisible ? "top-[117px]" : "top-[75px]"
                            } left-1/2 transform -translate-x-1/2 w-full 
    bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
    shadow-lg border-b-2 border-gray-300 dark:border-[#575662] 
    z-50 flex justify-center`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 py-6">
                              <div className="flex flex-col md:flex-row justify-between items-start gap-12">
                                <div className="space-y-12">
                                  <div className="max-w-[285px] space-y-2 mb-0">
                                    <h3 className="font-instrument-sans font-medium text-[18px] leading-[20px] text-gray-900 dark:text-[#eeeef0]">
                                      Get Started For Free
                                    </h3>
                                    <p className="font-body-medium text-[#51515E] dark:text-[#eeeef0]">
                                      Streamline your Business with our Apps.
                                      Effortlessly for every team.
                                    </p>
                                  </div>
                                  <Button
                                    style={{ marginTop: "20px" }}
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                  >
                                    See How It Works
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>

                                <div className="flex flex-wrap gap-x-14 gap-y-8 text-xs sm:text-sm">
                                  {footerColumns.map((column, index) => (
                                    <div
                                      key={index}
                                      className="flex flex-col gap-5"
                                    >
                                      <h4 className="font-label-medium text-gray-900 dark:text-white">
                                        {column.title}
                                      </h4>
                                      <ul className="flex flex-col gap-4">
                                        {column.links.map((link, linkIndex) => (
                                          <li key={linkIndex}>
                                            <div className="flex items-start gap-3">
                                              <Link
                                                href={link.url}
                                                target={
                                                  link.url.startsWith("http")
                                                    ? "_blank"
                                                    : "_self"
                                                }
                                                rel={
                                                  link.url.startsWith("http")
                                                    ? "noopener noreferrer"
                                                    : undefined
                                                }
                                                className="font-body-medium text-[#51515E] dark:text-[#d7d7de] hover:text-gray-900 dark:hover:text-white transition-colors whitespace-nowrap flex items-center gap-2"
                                              >
                                                {link.text}
                                                {column.title === "Social" && (
                                                  <FaArrowRight className="w-2 h-2" />
                                                )}
                                              </Link>

                                              {link.isNew && (
                                                <div className="hidden md:flex items-center gap-2">
                                                  <div className="w-4 h-4 flex items-center justify-center">
                                                    <div className="w-[13px] h-[13px] bg-[#c3e6ff] rounded-[6.67px] flex items-center justify-center">
                                                      <div className="w-1 h-1 bg-[#286efb] rounded-sm" />
                                                    </div>
                                                  </div>
                                                  <span className="font-body-medium text-gray-900 dark:text-white whitespace-nowrap">
                                                    New
                                                  </span>
                                                </div>
                                              )}
                                            </div>
                                          </li>
                                        ))}
                                      </ul>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-4">
                    <div className="hidden md:flex items-center gap-4">
                      <div className="w-6">
                        <ThemeSwitch />
                      </div>
                      <div className="flex items-center gap-2 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-1 shadow-sm">
                        <button
                          className="cursor-pointer px-3 py-1 text-sm font-medium text-gray-900 dark:text-white"
                          onClick={() => {
                            window.location.replace(
                              `https://worksuite.clikkle.com/login?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Login
                        </button>

                        <button
                          className="cursor-pointer flex items-center gap-1 rounded-xl bg-[#286efb] hover:bg-[#3367d6] px-4 py-1.5 text-sm font-medium text-white transition"
                          onClick={() => {
                            window.location.replace(
                              `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Try for free
                          <span>
                            <ArrowRightIcon className="w-4" />
                          </span>
                        </button>
                      </div>
                    </div>
                  </div>
                </nav>
              </div>
            </div>
          </section>
        </section>
        <div
          className={`bg-white dark:bg-[#1a1a1a] border-b ${
            theme === "dark" ? "border-gray-700" : "border-gray-200"
          }`}
        >
          <div
            className={`max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 ${
              announcementOpen ? "pt-6" : ""
            } py-16 pt-4`}
          >
            <div className="text-center">
              <h1
                className={`text-4xl md:text-5xl font-bold text-[#286efb] dark:text-[#A9C5FF] mb-4`}
              >
                Our Blogs
              </h1>
              <p
                className={`text-xl ${
                  theme === "dark" ? "text-gray-300" : "text-gray-600"
                } max-w-3xl mx-auto`}
              >
                Stay updated with the latest insights, trends, and news from the
                world of business technology and innovation.
              </p>
            </div>
          </div>
        </div>

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <div className="mb-8 space-y-4 lg:space-y-0 lg:flex lg:items-center lg:justify-between">
            <div className="relative flex-1 max-w-md">
              <Search
                className={`absolute left-3 top-1/2 transform -translate-y-1/2 ${
                  theme === "dark" ? "text-gray-400" : "text-gray-500"
                } h-5 w-5`}
              />
              <input
                type="text"
                placeholder="Search blog posts..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className={`w-full pl-10 pr-4 py-3 rounded-lg border bg-white dark:bg-[#1a1a1a] ${
                  theme === "dark"
                    ? " border-gray-600 text-white placeholder-gray-400"
                    : "bg-white border-gray-300 text-gray-900 placeholder-gray-500"
                } focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
              />
            </div>

            <div className="flex flex-wrap gap-4 lg:ml-6">
              <div className="flex items-center gap-2">
                <Filter
                  className={`h-5 w-5 ${
                    theme === "dark" ? "text-gray-400" : "text-gray-500"
                  }`}
                />
                <select
                  value={selectedCategory}
                  onChange={(e) => setSelectedCategory(e.target.value)}
                  className={`px-4 py-2 rounded-lg border bg-white dark:bg-[#1a1a1a] ${
                    theme === "dark"
                      ? " border-gray-600 text-white"
                      : "bg-white border-gray-300 text-gray-900"
                  } focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                >
                  {blogCategories.map((category) => (
                    <option key={category} value={category}>
                      {category}
                    </option>
                  ))}
                </select>
              </div>

              <div className="flex items-center gap-2">
                <button
                  onClick={() => {
                    if (sortBy === "date") setSortBy("title");
                    else if (sortBy === "title") setSortBy("order");
                    else setSortBy("date");
                  }}
                  className={`flex items-center gap-2 px-4 py-2 rounded-lg border ${
                    theme === "dark"
                      ? " border-gray-600 text-white hover:bg-gray-700"
                      : "bg-white border-gray-300 text-gray-900 hover:bg-gray-50"
                  } transition-colors`}
                >
                  {sortBy === "date"
                    ? "Date"
                    : sortBy === "order"
                    ? "Latest"
                    : "Title"}
                  {sortOrder === "desc" ? (
                    <SortDesc className="h-4 w-4" />
                  ) : (
                    <SortAsc className="h-4 w-4" />
                  )}
                </button>
                <button
                  onClick={() =>
                    setSortOrder(sortOrder === "desc" ? "asc" : "desc")
                  }
                  className={`p-2 rounded-lg border ${
                    theme === "dark"
                      ? " border-gray-600 text-white hover:bg-gray-700"
                      : "bg-white border-gray-300 text-gray-900 hover:bg-gray-50"
                  } transition-colors`}
                >
                  {sortOrder === "desc" ? (
                    <SortDesc className="h-4 w-4" />
                  ) : (
                    <SortAsc className="h-4 w-4" />
                  )}
                </button>
              </div>
            </div>
          </div>

          <div className="mb-6">
            <p
              className={`text-sm ${
                theme === "dark" ? "text-gray-400" : "text-gray-600"
              }`}
            >
              Showing {posts.length} of {totalCount} posts
              {selectedCategory !== "All" && ` in "${selectedCategory}"`}
              {searchTerm && ` matching "${searchTerm}"`}
            </p>
          </div>

          {loading && currentPage === 1 ? (
            <div className="flex items-center justify-center bg-white dark:bg-[#1a1a1a]">
                  <div className="text-center py-16">
                    <div className="flex justify-center mb-6">
                      <div className="relative w-16 h-16">
                        <svg className="animate-spin" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                          <circle 
                            cx="32" 
                            cy="32" 
                            r="28" 
                            stroke="currentColor" 
                            strokeWidth="4"
                            strokeDasharray="8 8"
                            strokeLinecap="round"
                            className="text-gray-200 dark:text-[#2d2b3d] opacity-50"
                          />
                          <circle 
                            cx="32" 
                            cy="32" 
                            r="28" 
                            stroke="currentColor" 
                            strokeWidth="4"
                            strokeDasharray="44 132"
                            strokeLinecap="round"
                            className="text-blue-500 dark:text-blue-400"
                            transform="rotate(-90 32 32)"
                          />
                        </svg>
                      </div>
                    </div>

                    <p className="text-[#1a1a1a] dark:text-gray-100 font-semibold text-lg mb-2">
                      Loading blog posts...
                    </p>
                    <p className="text-gray-500 dark:text-gray-400 text-sm">
                      Please wait while we fetch the latest content
                    </p>
                  </div>
                </div>
          ) : posts.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
              {posts.map((post) => (
                <Card
                  key={post.id}
                  className={`overflow-hidden hover:shadow-lg transition-shadow duration-300  border border-gray-300 dark:border-[#575662]`}
                >
                  <div className="aspect-video overflow-hidden">
                    <img
                      src={post.featuredImage}
                      alt={post.title}
                      className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                    />
                  </div>

                  <CardContent className="p-6">
                    <div className="mb-3">
                      <span
                        className={`inline-block px-3 py-1 text-xs font-medium rounded-full ${
                          theme === "dark"
                            ? "bg-blue-900 text-blue-200"
                            : "bg-blue-100 text-blue-800"
                        }`}
                      >
                        {post.category}
                      </span>
                    </div>

                    <h3
                      className={`text-xl font-bold mb-3 line-clamp-2 ${
                        theme === "dark" ? "text-white" : "text-gray-900"
                      }`}
                    >
                      {post.title}
                    </h3>

                    <p
                      className={`text-sm mb-4 line-clamp-3 ${
                        theme === "dark" ? "text-gray-300" : "text-gray-600"
                      }`}
                    >
                      {post.excerpt}
                    </p>

                    <div
                      className={`flex items-center justify-between text-xs mb-4 ${
                        theme === "dark" ? "text-gray-400" : "text-gray-500"
                      }`}
                    >
                      <div className="flex items-center gap-4">
                        <div className="flex items-center gap-1">
                          <Calendar className="h-3 w-3" />
                          <span>{formatDate(post.publishedDate)}</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Clock className="h-3 w-3" />
                          <span>{post.readTime} min read</span>
                        </div>
                      </div>
                    </div>

                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <img
                          src={post.author.avatar}
                          alt={post.author.name}
                          className="w-8 h-8 rounded-full object-cover"
                        />
                        <span
                          className={`text-sm font-medium ${
                            theme === "dark" ? "text-gray-300" : "text-gray-700"
                          }`}
                        >
                          {post.author.name}
                        </span>
                      </div>

                      <Link
                        href={`/blogs/${post.slug}`}
                        className="inline-flex gap-1  items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors"
                      >
                        Read More
                        <ArrowRightIcon className="w-4" />
                      </Link>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="text-center py-12">
              <div
                className={`text-6xl mb-4 ${
                  theme === "dark" ? "text-gray-600" : "text-gray-400"
                }`}
              >
                📝
              </div>
              <h3
                className={`text-xl font-semibold mb-2 ${
                  theme === "dark" ? "text-white" : "text-gray-900"
                }`}
              >
                No blog posts found
              </h3>
              <p
                className={`${
                  theme === "dark" ? "text-gray-400" : "text-gray-600"
                }`}
              >
                Try adjusting your search terms or filters to find what
                you&apos;re looking for.
              </p>
              {error && <p className="mt-2 text-red-600">{error}</p>}
            </div>
          )}

          {currentPage < totalPages && (
            <div className="flex items-center justify-center gap-2 mt-8">
              <Button
                variant="outline"
                onClick={() => setCurrentPage((prev) => prev + 1)}
                disabled={loading}
                className={`min-w-[120px] cursor-pointer bg-white dark:bg-[#1a1a1a] ${
                  theme === "dark"
                    ? "border-gray-600 text-gray-300 hover:bg-gray-800"
                    : ""
                }`}
              >
                <FaArrowCircleDown />
                {loading ? "Loading..." : "Read More"}
              </Button>
            </div>
          )}
        </div>
      </div>

      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
      <Footer />
    </>
  );
};

export default BlogsHome;
